--[[
	ExtendedMissionInfo.lua
	
	Autor: 		Ifko[nator]
	Datum: 		31.12.2024
	Version: 	1.0
	
	Changelog:	v1.0 @31.12.2024 - initial implementation in FS 25
]]

ExtendedMissionInfo = {};

ExtendedMissionInfo.currentModName = "";

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "EXTENDED MISSION INFO" or mod.title:upper() == "ERWEITERTE MISSION INFO" then		
		if g_modIsLoaded[tostring(mod.modName)] then	
			ExtendedMissionInfo.currentModName = mod.modName;
			
			break;
		end;
	end;
end;

MissionManager.MAX_MISSIONS = 60;
MissionManager.MAX_MISSIONS_PER_FARM = 10;

function ExtendedMissionInfo:loadMapData()
    if #g_missionManager.missionTypes ~= 0 then
        for _, missionType in ipairs(g_missionManager.missionTypes) do
            missionType.data.maxNumInstances = 8;
        end;
    end;
end;

g_missionManager.loadMapData = Utils.appendedFunction(MissionManager.loadMapData, ExtendedMissionInfo.loadMapData)

local function getNewContractLimitedReachedText(contractLimitedReachedText)
    local wordToSearch = "three";
    local wordToReplace = tostring(MissionManager.MAX_MISSIONS_PER_FARM);

    if g_languageShort == "br" or g_languageShort == "pt" then
        wordToSearch = "três";
    elseif g_languageShort == "cs" or g_languageShort == "ct" then
        wordToSearch = "三";
        wordToReplace = "十";
    elseif g_languageShort == "cz" then
        wordToSearch = "tři";
    elseif g_languageShort == "da" or g_languageShort == "it" or g_languageShort == "no" or g_languageShort == "sv" then
        wordToSearch = "tre";
    elseif g_languageShort == "de" then
        wordToSearch = "drei";
    elseif g_languageShort == "ea" or g_languageShort == "es" then
        wordToSearch = "tres";
    elseif g_languageShort == "fc" or g_languageShort == "fr" then
        wordToSearch = "trois";
    elseif g_languageShort == "fi" then
        wordToSearch = "kolme";
    elseif g_languageShort == "hu" then
        wordToSearch = "három";
    elseif g_languageShort == "id" then
        wordToSearch = "tiga";
    elseif g_languageShort == "jp" or g_languageShort == "pl" or g_languageShort == "ru" then
        wordToSearch = "3";
    elseif g_languageShort == "kr" then
        wordToSearch = "세";
    elseif g_languageShort == "nl" then
        wordToSearch = "drie";
    elseif g_languageShort == "ro" then
        wordToSearch = "trei";
    elseif g_languageShort == "tr" then
        wordToSearch = "üç tane";
    elseif g_languageShort == "uk" then
        wordToSearch = "три";
    elseif g_languageShort == "vi" then
        wordToSearch = "ba";
    end;

   return contractLimitedReachedText:gsub(wordToSearch, wordToReplace);
end;

local function getUseFillTypeTitle(fruitType, title)
    local harvestTitleCheck = g_i18n:getText("contract_field_harvest_title");
	local sowTitleCheck = g_i18n:getText("contract_field_sow_title");
	local fertilizeTitleCheck = g_i18n:getText("contract_field_fertilize_title");
	local herbicideTitleCheck = g_i18n:getText("contract_field_herbicide_title");
	local hoeTitleCheck = g_i18n:getText("contract_field_hoe_title");
	local weedTitleCheck = g_i18n:getText("contract_field_weed_title");
    
    if fruitType == nil or fruitType.fillType == nil then
        return false;
    end;

    return title == harvestTitleCheck or title == sowTitleCheck or title == fertilizeTitleCheck or title == herbicideTitleCheck or title == hoeTitleCheck or title == weedTitleCheck;
end;

function ExtendedMissionInfo:onMissionStarted(superFunc, missionState, leaseVehicles)
	g_messageCenter:unsubscribe(MissionStartEvent, self);
	
    if missionState == MissionStartState.OK then
		if leaseVehicles then
			InfoDialog.show(g_i18n:getText("contract_vehiclesAtShop"), nil, nil, DialogElement.TYPE_INFO);
		else
			InfoDialog.show(g_i18n:getText("contract_started"), nil, nil, DialogElement.TYPE_INFO);
		end;
	elseif missionState == MissionStartState.LIMIT_REACHED then
		InfoDialog.show(getNewContractLimitedReachedText(g_i18n:getText("contract_limitedReached")), nil, nil, DialogElement.TYPE_WARNING);

		return;
	elseif missionState == MissionStartState.ALREADY_STARTED then
		InfoDialog.show(g_i18n:getText("contract_alreadyStarted"), nil, nil, DialogElement.TYPE_WARNING);

		return;
	elseif missionState == MissionStartState.NOT_AVAILABLE_ANYMORE then
		InfoDialog.show(g_i18n:getText("contract_notAvailableAnymore"), nil, nil, DialogElement.TYPE_WARNING);

		return;
	elseif missionState == MissionStartState.NO_ACCESS then
		InfoDialog.show(g_i18n:getText("contract_noAccess"), nil, nil, DialogElement.TYPE_WARNING);

		return;
	elseif missionState == MissionStartState.CANNOT_BE_STARTED_NOW then
		InfoDialog.show(g_i18n:getText("contract_cannotBeStartedNow"), nil, nil, DialogElement.TYPE_WARNING);

		return;
	elseif missionState == MissionStartState.PENDING_MISSION then
		InfoDialog.show(g_i18n:getText("contract_pendingMissionStart"), nil, nil, DialogElement.TYPE_WARNING);

		return;
	elseif missionState == MissionStartState.NO_PERMISSION then
		InfoDialog.show(g_i18n:getText("contract_noPermission"), nil, nil, DialogElement.TYPE_WARNING);
	else
		InfoDialog.show(g_i18n:getText("contract_startFailed"), nil, nil, DialogElement.TYPE_WARNING);
	end;
end;

InGameMenuContractsFrame.onMissionStarted = Utils.overwrittenFunction(InGameMenuContractsFrame.onMissionStarted, ExtendedMissionInfo.onMissionStarted);

function ExtendedMissionInfo:setField(superFunc, field)
   	self.field = field;

    field:setMission(self);

    local fieldId = field:getId();

    if fieldId ~= nil then
		local fruitTypeTitle = "";
        
        local fruitType = g_fruitTypeManager:getFruitTypeByIndex(field.fieldState.fruitTypeIndex);
       
		if getUseFillTypeTitle(fruitType, self.title) then
			fruitTypeTitle = fruitType.fillType.title .. " ";
		end;

		self.progressTitle = ("%s%s"):format(fruitTypeTitle, self.title);
		self.fieldId = fieldId;
    end;

    self.fieldName = field:getName();

    g_fieldManager:onFieldMissionStarted();

    self:createMapHotspot();
end;

AbstractFieldMission.setField = Utils.overwrittenFunction(AbstractFieldMission.setField, ExtendedMissionInfo.setField);

function ExtendedMissionInfo:update(superFunc, dt)
    local mission = g_currentMission;

    if self.pendingVehicleUniqueIds ~= nil then
        for i = #self.pendingVehicleUniqueIds, 1, -1 do
            local uniqueId = self.pendingVehicleUniqueIds[i];
            local vehicle = mission.vehicleSystem:getVehicleByUniqueId(uniqueId);

            if vehicle ~= nil then
                table.remove(self.pendingVehicleUniqueIds, i);
                table.insert(self.vehicles, vehicle);
            end;
        end;

        if #self.pendingVehicleUniqueIds == 0 then
            self.pendingVehicleUniqueIds = nil;
        end;
    end;

    if self.tryToAddMissingVehicles and self.pendingVehicleUniqueIds == nil then
        if #self.vehicles ~= #self.vehiclesToLoad then
            for _, info in ipairs(self.vehiclesToLoad) do
                local found = false;

                for _, vehicle in ipairs(self.vehicles) do
                    if info.filename == vehicle.configFileName then
                        found = true;

                        break;
                    end;
                end;

                if not found then
                    self:spawnVehicle(info);
                end;
            end;
        end;

        self.tryToAddMissingVehicles = false;
    end;

    if self.isServer then
        if self.status == MissionStatus.PREPARING then
            if self:getIsPrepared() then
                self:finishedPreparing();
            end;
        end;
    end;

    if self.status == MissionStatus.RUNNING then
        if self.isServer then
            if self:isTimedOut() then
                self:finish(MissionFinishState.TIMED_OUT);
            elseif not self:validate() then
                self:finish(MissionFinishState.FAILED);
            end;
        end;

        local missionList = g_missionManager:getMissions(g_currentMission:getFarmId());
        local sellingStationName = nil; 
        local textExtension = nil;
        local baleTypeTitle = nil;
        local baleTypeText = nil;
        local numLeftTrees = nil;
        local deliveredTitle = nil;
        local deliveredText = nil;
		
        if self.progressBar == nil then
            local contractDetails = g_i18n:getText("contract_details_field") .. " ";
            local fieldId = self.fieldId;

            if fieldId == nil then
                contractDetails = self:getLocation();

                fieldId = "";
            end;
 
            self.progressBar = mission.hud:addSideNotificationProgressBar(("%s%s"):format(g_i18n:getText("contract_title") .. ", " .. contractDetails, fieldId), self.progressTitle, self.completion);
        end;

        for _, mission in ipairs(missionList) do
            if mission:getIsInProgress() and self.title == mission.title and self:getLocation() == mission:getLocation() then
                if self.title == g_i18n:getText("contract_field_harvest_title") or self.title == g_i18n:getText("contract_forestry_treeTransport_title") then
                    if mission.sellingStation ~= nil and mission.sellingStation.getName ~= nil then
                        sellingStationName = mission.sellingStation:getName();
                    end;

                    if mission.numTrees ~= nil then
                        local numLeftTrees = mission.numTrees - mission.numDeletedTrees;
                        
                        local text = g_i18n:getText("contract_forestry_deadwood_remainingTrees"):format(numLeftTrees);
                        
                        if numLeftTrees == 1 then
                            text = g_i18n:getText("contract_forestry_deadwood_oneRemainingTree");
                        end;
                        
                        deliveredTitle = g_i18n:getText("contract_forestry_treeTransport_details_deliveredNumTrees");
                        deliveredText = tostring(mission.numDeliveredTrees) .. ", (" .. text .. ")";
                    end;
                elseif self.title == g_i18n:getText("contract_forestry_deadwood_title") then
                    local numLeftCutDownTrees = mission.numDeadTrees - mission.numCutDownTrees;

                    textExtension = g_i18n:getText("contract_forestry_deadwood_remainingTrees"):format(numLeftCutDownTrees);

                    if numLeftCutDownTrees == 1 then
                        textExtension = g_i18n:getText("contract_forestry_deadwood_oneRemainingTree");
                    end;
                elseif self.title == g_i18n:getText("contract_map_destructibleRock_title") then
                    local numLeftRocks = #mission.rocksAll - mission.numRocksDestroyed;

                    textExtension = g_i18n:getText("contract_map_destructibleRock_remainingRocks"):format(numLeftRocks);
            
                    if numLeftRocks == 1 then
                        textExtension = g_i18n:getText("contract_map_destructibleRock_oneRemainingRock");
                    end;
                elseif self.title == g_i18n:getText("contract_field_bale_title") then
                    baleTypeTitle = g_i18n:getText("contract_details_bale_type");
                    baleTypeText = mission.needRoundbaler and g_i18n:getText("contract_details_bale_type_round") or g_i18n:getText("contract_details_bale_type_square");
                elseif self.title == g_i18n:getText("contract_field_baleWrap_title") then
                    baleTypeTitle = g_i18n:getText("contract_details_bale_type");
                    baleTypeText = g_baleManager:getIsRoundBale(mission.baleTypeIndex) and g_i18n:getText("contract_details_bale_type_round") or g_i18n:getText("contract_details_bale_type_square");
                    
                    local numBalesWrapped = 0;

                    for _, bale in ipairs(mission.bales) do
                        if not bale:getIsMounted() then
                            numBalesWrapped = numBalesWrapped + bale.wrappingState
                        end
                    end

                    local numBalesLeft = mission.numOfBales - numBalesWrapped;
                    
                    textExtension = g_i18n:getText("contract_baleWrap_remainingBales", ExtendedMissionInfo.currentModName):format(numBalesLeft);

                    if numBalesLeft == 1 then
                        textExtension = g_i18n:getText("contract_baleWrap_oneRemainingBale", ExtendedMissionInfo.currentModName);
                    end;
                end;
            end;
        end;
        
        self.progressBar.progress = self.completion;
        self.progressBar.sellingStationName = sellingStationName;
        self.progressBar.baleTypeTitle = baleTypeTitle;
        self.progressBar.baleTypeText = baleTypeText;
        self.progressBar.textExtension = textExtension;
        self.progressBar.deliveredTitle = deliveredTitle;
        self.progressBar.deliveredText = deliveredText;
        
        if g_localPlayer ~= nil and g_localPlayer.farmId == self.farmId then
            mission.hud:markSideNotificationProgressBarForDrawing(self.progressBar);
        end;
    end;

    if self.status == MissionStatus.RUNNING or self.status == MissionStatus.PREPARING then
        self:raiseActive();
    end;
end;

AbstractMission.update = Utils.overwrittenFunction(AbstractMission.update, ExtendedMissionInfo.update);

function ExtendedMissionInfo:storeScaledValues()
	self.textMaxWidth = self:scalePixelToScreenWidth(400);
end;

SideNotification.storeScaledValues = Utils.appendedFunction(SideNotification.storeScaledValues, ExtendedMissionInfo.storeScaledValues);

function ExtendedMissionInfo:draw(superFunc)
    SideNotification:superClass().draw(self);

    local posX, posY = self:getPosition();

    setTextBold(true);
    setTextAlignment(RenderText.ALIGN_LEFT);
    setTextColor(1, 1, 1, 1);

    local barBgColor = HUD.COLOR.BACKGROUND_DARK;
    local activeColor = HUD.COLOR.ACTIVE;

    local hasProgressBars = false;
    local textMaxWidth = self.textMaxWidth;

    for _, progressBar in ipairs(self.progressBars) do
        if progressBar.isVisible then
            posY = posY - self.notificationOffsetY;

            local progress = math.clamp(progressBar.progress, 0, 1);

            local title = "";

            if progressBar.title ~= nil then
                title = utf8ToUpper(progressBar.title .. (progressBar.text ~= nil and ": " or ""));
            end;

            local text = "";

            if progressBar.text ~= nil then
                if progressBar.textExtension ~= nil then
                    if progressBar.textSaved == nil then
                        progressBar.textSaved = progressBar.text;
                    end;
                    
                    progressBar.text = progressBar.textSaved .. ", (" .. tostring(progressBar.textExtension) .. ")";
                end;
                
                text = utf8ToUpper(progressBar.text);
            end;

            setTextBold(true);
            setTextAlignment(RenderText.ALIGN_LEFT);

            local titleWidth = getTextWidth(self.progressBarTextSize, title);

            textMaxWidth = self.textMaxWidth - titleWidth;

            setTextWrapWidth(textMaxWidth);
            
            local textPosX = self.progressBarBgTop.x + self.progressBarTitleOffsetX;
            local textPosY = posY + self.progressBarTitleOffsetY;

            renderText(textPosX, textPosY, self.progressBarTextSize, title);

            setTextBold(false);

            renderText(textPosX + titleWidth, textPosY, self.progressBarTextSize, text);

            setTextWrapWidth(0);

            local textHeight = getTextHeight(self.progressBarTextSize, text);
            local totalHeight = textHeight + self.textOffsetY * 2 + self.barHeight;

            local additionalTitle = nil;
            local additionalText = nil;

            if progressBar.sellingStationName ~= nil then
                additionalTitle = g_i18n:getText("contract_details_harvesting_sellingStation");
                additionalText = tostring(progressBar.sellingStationName);
            elseif progressBar.baleTypeTitle ~= nil and progressBar.baleTypeText ~= nil then
                additionalTitle = tostring(progressBar.baleTypeTitle);
                additionalText = tostring(progressBar.baleTypeText);
            end;

            if additionalTitle ~= nil and additionalText ~= nil then
                textPosY = textPosY - self.progressBarTextSize;
                
                local additionalTitle = utf8ToUpper(additionalTitle .. ": ") .. " "; --## don't ask me, why i need here an extra space charachter, i don't know... But if it works, it works...
                local additionalText =  utf8ToUpper(additionalText);

                local additionalTitleWidth = getTextWidth(self.progressBarTextSize, additionalTitle);

                textMaxWidth = self.textMaxWidth - additionalTitleWidth;

                setTextWrapWidth(textMaxWidth);

                setTextBold(true);

                renderText(textPosX, textPosY, self.progressBarTextSize, additionalTitle);

                setTextBold(false);

                renderText(textPosX + additionalTitleWidth, textPosY, self.progressBarTextSize, additionalText);

                setTextWrapWidth(0);

                totalHeight = totalHeight + textHeight;
            end;

            if progressBar.deliveredTitle ~= nil and progressBar.deliveredText ~= nil then
                textPosY = textPosY - self.progressBarTextSize;
                
                local deliveredTitle = utf8ToUpper(progressBar.deliveredTitle .. ": ") .. " "; --## don't ask me, why i need here an extra space charachter, i don't know... But if it works, it works...
                local deliveredText = utf8ToUpper(tostring(progressBar.deliveredText));

                local deliveredTitleWidth = getTextWidth(self.progressBarTextSize, deliveredTitle);

                textMaxWidth = self.textMaxWidth - deliveredTitleWidth;

                setTextWrapWidth(textMaxWidth);

                setTextBold(true);

                renderText(textPosX, textPosY, self.progressBarTextSize, deliveredTitle);

                setTextBold(false);

                renderText(textPosX + deliveredTitleWidth, textPosY, self.progressBarTextSize, deliveredText);

                setTextWrapWidth(0);

                totalHeight = totalHeight + textHeight;
            end;

            self.progressBarBgScale:setDimension(nil, totalHeight - self.progressBarBgTop.height - self.progressBarBgBottom.height);

            self.progressBarBgTop:setPosition(nil, posY - self.progressBarBgTop.height);
            self.progressBarBgTop:render();
            self.progressBarBgScale:setPosition(nil, self.progressBarBgTop.y - self.progressBarBgScale.height);
            self.progressBarBgScale:render();
            self.progressBarBgBottom:setPosition(nil, self.progressBarBgScale.y - self.progressBarBgBottom.height);
            self.progressBarBgBottom:render();

            self.bar:setColor(barBgColor[1], barBgColor[2], barBgColor[3], barBgColor[4]);
            self.bar:setMiddlePart(nil, self.barMaxScaleWidth, nil);
            self.bar:setPosition(self.progressBarBgBottom.x + self.barOffsetX, self.progressBarBgBottom.y + self.barOffsetY);
            self.bar:render();

            if progress >= 0.01 then
                self.bar:setColor(activeColor[1], activeColor[2], activeColor[3], activeColor[4]);
                self.bar:setMiddlePart(nil, self.barMaxScaleWidth * progress, nil);
                self.bar:setPosition(self.progressBarBgBottom.x + self.barOffsetX, self.progressBarBgBottom.y + self.barOffsetY);
                self.bar:render();
            end;

            setTextAlignment(RenderText.ALIGN_RIGHT);

            local progressText = string.format("%d%%", progress * 100);

            renderText(posX + self.progressBarProgressTextOffsetX, self.progressBarBgBottom.y + self.progressBarProgressTextOffsetY, self.progressBarProgressTextSize, progressText);

            setTextAlignment(RenderText.ALIGN_LEFT);

            hasProgressBars = true;

            posY = posY - totalHeight;
        end;

        progressBar.isVisible = false;
    end;

    if hasProgressBars then
        posY = posY - self.progressBarSectionOffsetY;
    end;

    setTextBold(true);
    setTextAlignment(RenderText.ALIGN_RIGHT);

    for notificationIndex = 1, math.min(#self.notificationQueue, SideNotification.MAX_NOTIFICATIONS) do
        posY = posY - self.bgRight.height - self.notificationOffsetY;

        local notification = self.notificationQueue[notificationIndex];

        local textWidth = getTextWidth(self.textSize, notification.text);
        local scaleWidth = textWidth;

        self.bgRight:setPosition(posX - self.bgRight.width, posY);
        self.bgRight:render();
        self.bgScale:setDimension(scaleWidth, nil);
        self.bgScale:setPosition(self.bgRight.x - self.bgScale.width, posY);
        self.bgScale:render();
        self.bgLeft:setPosition(self.bgScale.x - self.bgLeft.width, posY);
        self.bgLeft:render();

        setTextColor(notification.color[1], notification.color[2], notification.color[3], notification.color[4]);
        renderText(self.bgRight.x, self.bgRight.y + self.textOffsetY, self.textSize, notification.text);
    end;

    setTextAlignment(RenderText.ALIGN_LEFT);
    setTextColor(1, 1, 1, 1);
    setTextBold(false);
end;

SideNotification.draw = Utils.overwrittenFunction(SideNotification.draw, ExtendedMissionInfo.draw);